<?php

namespace AmeliaBooking\Application\Services\Coupon;

use AmeliaBooking\Domain\Common\Exceptions\CouponInvalidException;
use AmeliaBooking\Domain\Common\Exceptions\CouponUnknownException;
use AmeliaBooking\Domain\Entity\Bookable\Service\Service;
use AmeliaBooking\Domain\Entity\Coupon\Coupon;
use AmeliaBooking\Infrastructure\Common\Exceptions\QueryExecutionException;
use AmeliaBooking\Domain\Common\Exceptions\InvalidArgumentException;
use AmeliaBooking\Domain\ValueObjects\Number\Integer\Id;
use AmeliaBooking\Infrastructure\Common\Container;
use AmeliaBooking\Infrastructure\Repository\Booking\Appointment\AppointmentRepository;
use AmeliaBooking\Infrastructure\Repository\Booking\Appointment\CustomerBookingRepository;
use AmeliaBooking\Infrastructure\Repository\Coupon\CouponRepository;
use AmeliaBooking\Infrastructure\Repository\Coupon\CouponServiceRepository;
use AmeliaBooking\Infrastructure\WP\Translations\FrontendStrings;

/**
 * Class CouponApplicationService
 *
 * @package AmeliaBooking\Application\Services\Coupon
 */
class CouponApplicationService
{
    private $container;

    /**
     * CouponApplicationService constructor.
     *
     * @param Container $container
     *
     * @throws \InvalidArgumentException
     */
    public function __construct(Container $container)
    {
        $this->container = $container;
    }


    /**
     * @param Coupon $coupon
     *
     * @return boolean
     *
     * @throws \Slim\Exception\ContainerValueNotFoundException
     * @throws InvalidArgumentException
     * @throws QueryExecutionException
     * @throws \Interop\Container\Exception\ContainerException
     */
    public function add($coupon)
    {
        /** @var CouponRepository $couponRepository */
        $couponRepository = $this->container->get('domain.coupon.repository');

        /** @var CouponServiceRepository $couponServiceRepository */
        $couponServiceRepo = $this->container->get('domain.coupon.service.repository');

        $couponId = $couponRepository->add($coupon);

        $coupon->setId(new Id($couponId));

        /**
         * Add coupon services
         */
        foreach ((array)$coupon->getServiceList()->keys() as $key) {
            if (!($service = $coupon->getServiceList()->getItem($key)) instanceof Service) {
                throw new InvalidArgumentException('Unknown type');
            }

            $couponServiceRepo->add($coupon, $service);
        }

        return $couponId;
    }

    /**
     * @param Coupon $oldCoupon
     * @param Coupon $newCoupon
     *
     * @return boolean
     *
     * @throws \Slim\Exception\ContainerValueNotFoundException
     * @throws InvalidArgumentException
     * @throws QueryExecutionException
     * @throws \Interop\Container\Exception\ContainerException
     */
    public function update($oldCoupon, $newCoupon)
    {
        /** @var CouponRepository $couponRepository */
        $couponRepository = $this->container->get('domain.coupon.repository');

        /** @var CouponServiceRepository $couponServiceRepository */
        $couponServiceRepo = $this->container->get('domain.coupon.service.repository');

        /**
         * Update coupon
         */
        $couponRepository->update($oldCoupon->getId()->getValue(), $newCoupon);


        $oldServiceIds = [];
        $newServiceIds = [];

        foreach ((array)$newCoupon->getServiceList()->keys() as $serviceKey) {
            if (!($newService = $newCoupon->getServiceList()->getItem($serviceKey)) instanceof Service) {
                throw new InvalidArgumentException('Unknown type');
            }

            $newServiceIds[] = $newService->getId()->getValue();
        }

        foreach ((array)$oldCoupon->getServiceList()->keys() as $serviceKey) {
            if (!($oldService = $oldCoupon->getServiceList()->getItem($serviceKey)) instanceof Service) {
                throw new InvalidArgumentException('Unknown type');
            }

            $oldServiceIds[] = $oldService->getId()->getValue();
        }


        /**
         * Manage coupon services
         */
        foreach ((array)$newCoupon->getServiceList()->keys() as $key) {
            if (!($newService = $newCoupon->getServiceList()->getItem($key)) instanceof Service) {
                throw new InvalidArgumentException('Unknown type');
            }

            if (!in_array($newService->getId()->getValue(), $oldServiceIds, true)) {
                $couponServiceRepo->add($newCoupon, $newService);
            }
        }

        foreach ((array)$oldCoupon->getServiceList()->keys() as $key) {
            if (!($oldService = $oldCoupon->getServiceList()->getItem($key)) instanceof Service) {
                throw new InvalidArgumentException('Unknown type');
            }

            if (!in_array($oldCoupon->getServiceList()->getItem($key)->getId()->getValue(), $newServiceIds, true)) {
                $couponServiceRepo->deleteForService($oldCoupon->getId()->getValue(), $oldService->getId()->getValue());
            }
        }

        return true;
    }

    /**
     * @param Coupon $coupon
     *
     * @return boolean
     *
     * @throws \Slim\Exception\ContainerValueNotFoundException
     * @throws QueryExecutionException
     * @throws \Interop\Container\Exception\ContainerException
     */
    public function delete($coupon)
    {
        /** @var CouponRepository $couponRepository */
        $couponRepository = $this->container->get('domain.coupon.repository');

        /** @var CouponServiceRepository $couponServiceRepository */
        $couponServiceRepository = $this->container->get('domain.coupon.service.repository');

        /** @var CouponServiceRepository $couponEventRepository */
        $couponEventRepository = $this->container->get('domain.coupon.event.repository');

        /** @var CustomerBookingRepository $customerBookingRepository */
        $customerBookingRepository = $this->container->get('domain.booking.customerBooking.repository');

        return $couponServiceRepository->deleteByEntityId($coupon->getId()->getValue(), 'couponId') &&
            $couponEventRepository->deleteByEntityId($coupon->getId()->getValue(), 'couponId') &&
            $customerBookingRepository->unsetByEntityId($coupon->getId()->getValue(), 'couponId') &&
            $couponRepository->delete($coupon->getId()->getValue());
    }

    /** @noinspection MoreThanThreeArgumentsInspection */
    /**
     * @param string $couponCode
     * @param int    $serviceId
     * @param int    $userId
     * @param bool   $inspectCoupon
     *
     * @return Coupon
     *
     * @throws \Slim\Exception\ContainerValueNotFoundException
     * @throws InvalidArgumentException
     * @throws QueryExecutionException
     * @throws \Interop\Container\Exception\ContainerException
     * @throws CouponUnknownException
     * @throws CouponInvalidException
     */
    public function processCoupon($couponCode, $serviceId, $userId, $inspectCoupon)
    {
        /** @var CouponRepository $couponRepository */
        $couponRepository = $this->container->get('domain.coupon.repository');

        $coupons = $couponRepository->getAllByCriteria([
            'code' => $couponCode
        ]);

        /** @var Coupon $coupon */
        $coupon = $coupons->length() ? $coupons->getItem($coupons->keys()[0]) : null;

        $this->inspectCoupon($coupon, $serviceId, $userId, $inspectCoupon);

        return $coupon;
    }

    /** @noinspection MoreThanThreeArgumentsInspection */
    /**
     * @param Coupon $coupon
     * @param int    $serviceId
     * @param int    $userId
     * @param bool   $inspectCoupon
     *
     * @return boolean
     *
     * @throws \Slim\Exception\ContainerValueNotFoundException
     * @throws \Interop\Container\Exception\ContainerException
     * @throws CouponUnknownException
     * @throws CouponInvalidException
     * @throws QueryExecutionException
     */
    public function inspectCoupon($coupon, $serviceId, $userId, $inspectCoupon)
    {
        if (!$coupon || ($serviceId !== null && !$coupon->getServiceList()->keyExists($serviceId))) {
            throw new CouponUnknownException(FrontendStrings::getCommonStrings()['coupon_unknown']);
        }

        if ($inspectCoupon && ($coupon->getStatus()->getValue() === 'hidden' ||
                ($coupon && $coupon->getUsed()->getValue() >= $coupon->getLimit()->getValue())
        )) {
            throw new CouponInvalidException(FrontendStrings::getCommonStrings()['coupon_invalid']);
        }

        if ($userId && $coupon->getCustomerLimit()->getValue() > 0) {
            /** @var AppointmentRepository $appointmentRepo */
            $appointmentRepo = $this->container->get('domain.booking.appointment.repository');

            $customerAppointments = $appointmentRepo->getFiltered([
                'customerId'      => $userId,
                'bookingCouponId' => $coupon->getId()->getValue()
            ]);

            if ($customerAppointments->length() >= $coupon->getCustomerLimit()->getValue()) {
                throw new CouponInvalidException(FrontendStrings::getCommonStrings()['coupon_invalid']);
            }
        }

        return true;
    }
}
